% ExtractMEAData_UserChosenElectrodes_Batch.m
%   Version: MATLAB 2019a, ADMINISTRATIVE RIGHTS
% 	Description:
% 	
% 		Extracts data from an xlsx file and exports the data into .mat 
%       files and excel spreadsheets. Selected electrode recordings from the
%       MEA are considered.
% 		
% 		Requires a correct file name convention. File MUST end with 
%                       Spikes.xlsx OR SPs.xlsx
% 		
% 		Performs SP grouping if necessary	
% 		
% 		.mat file OUTPUTS
% 			Spiketrains:
% 			Percent Matrix:
% 			RawCountMatrix:
% 		
% 		Excel Spreadsheet OUTPUTS
% 			Order of Electrodes:
% 			Coincidence Count Matrix:
% 		
% 	Users Setup:
% 		Folder Directory containing the xlsx file
% 		Coincidence Thresholds for Spikes and SP's
% 		SP grouping option & grouping parameters
% 		
% 	Function Requirements:
% 		[C1,C2,Deltas] = SpikeSynchrony(L1,L2,time,MaxDiff): Find
% 		coincidences between two spike trains
clear;
close all;
clc;
%% Users Setup:

%Enter the name of the folder containing the batch of excel files
BatchFolder = 'Batch4Processing';

%Choose which electrodes to consider in each group
ElectrodeGroup1 = [14,16,24,25,26];  %Place all electrode numbers in between square brackets (separated by commas)
ElectrodeGroup2 = [45,54,55,64,65];

%Set the coincidence thresholds for SPs and Spikes
CoincidenceThreshold_Spikes = 0.02;     %seconds

CoincidenceThreshold_SPs = 0.2;         %seconds

%Turn SP grouping on (1) or off (0)
SPgrouping = 1;
%Set the grouping parameters if on
MaximumSeparationInGroup = 1.00;    %the maximum allowable separation (in seconds) between timestamps before they are considered to be in separate groups
MaximumGroupSize = 3;               %the maximum number of timestamps allowed in a group

%% -----------------------------------------------------------------------

%find the name of each file
CSVfiles = dir(BatchFolder);

%Remove all files which aren't in .xlsx format
ignoreFiles = [];
for i = 1:length(CSVfiles)
    
    try
        fileformat = CSVfiles(i).name(end-3:end);
        if(~strcmp(fileformat,'xlsx'))
            ignoreFiles = [ignoreFiles i];
        end
    catch
        ignoreFiles = [ignoreFiles i];
    end
    
end
CSVfiles(ignoreFiles) = [];

%Step through extraction code for each file in turn
for i = 1:length(CSVfiles)
    CSVfile = CSVfiles(i).name;
    
    disp("Extracting from " + CSVfile)
    
    %Check which spike type
    SpikeTypeIndicator = CSVfile(end-6);
    switch SpikeTypeIndicator
        case 'P'
            SpikeType = "SPs";
            MaxDifference = CoincidenceThreshold_SPs;
            disp("File is an SP record")
        case 'e'
            SpikeType = "Spikes";
            MaxDifference = CoincidenceThreshold_Spikes;
            disp("File is a spike record")
        otherwise
            disp("Wrong file name convention, spike type not determined, skipping..")
            continue
    end
    
    SpikeTrains = [];
    MinimumSpikeNum = 1;    %minimum number of spikes for consideration

    cd(BatchFolder)
    [~,TextData,AllData] = xlsread(CSVfile,1,'A1:A50000');
    [~,~,Electrodes] = xlsread(CSVfile,1,'C1:C50000');
    Index = find(TextData=="t");
    cd ..
    
    TrainNum = 1;
    maxLat = 0;

    %Grab numerical data from each electrode in the file
    for j = 1:length(Index)-1
        try
            Latencies = AllData(Index(j)+3:Index(j+1)-2);
            if(j==length(Index)-1)
                Latencies = AllData(Index(j)+3:Index(j+1)-5);
            end
            
            if(length(Latencies)>=MinimumSpikeNum)
                SpikeTrains(TrainNum).Latencies = cell2mat(Latencies);
                SpikeTrains(TrainNum).Electrode = cell2mat(Electrodes(Index(j)));

                if(max(cell2mat(Latencies))>maxLat)
                    maxLat = max(cell2mat(Latencies));
                end

                TrainNum = TrainNum+1;
            end
        catch
            break;
        end
    end

    if(TrainNum==1)
        SpikeTrains.Latencies = [];
        SpikeTrains.Electrode = 12;
    end

    %% Perform SP grouping if appropriate
    
    if(strcmp(SpikeType,"SPs")&&(SPgrouping==1)&&~isempty(SpikeTrains(1).Latencies))
        for j = 1:length(SpikeTrains)
            Ungrouped = SpikeTrains(j).Latencies;
            Grouped = zeros(size(Ungrouped));
            Grouped(1) = Ungrouped(1);
            GroupCount = 1;
            GroupedCounter = 2;
            for k = 2:length(Ungrouped)
                if((Ungrouped(k)-Ungrouped(k-1))<MaximumSeparationInGroup)
                    if(GroupCount <= MaximumGroupSize)
                        GroupCount = GroupCount+1;
                    else
                        Grouped(GroupedCounter) = Ungrouped(k);
                        GroupedCounter = GroupedCounter+1;
                        GroupCount = 1;
                    end
                else
                    Grouped(GroupedCounter) = Ungrouped(k);
                    GroupedCounter = GroupedCounter+1;
                    GroupCount = 1;
                end
            end
            
            %Remove all the extra zeros
            Grouped = Grouped(Grouped>0);
            SpikeTrains(j).Latencies = Grouped;
        end
        
    end
    
    %% Filter out electrodes that aren't in the specified groups
    
    rejects = [];
    for j = 1:length(SpikeTrains)
        
        if(~ismember(SpikeTrains(j).Electrode,ElectrodeGroup1) && ~ismember(SpikeTrains(j).Electrode,ElectrodeGroup2))
            rejects = [rejects j];
        end
        
    end
    
    SpikeTrains(rejects) = [];
    
    fprintf('%d of the chosen electrodes have activity \n',length(SpikeTrains));
    
    %% Process coincident spike statistics

    PercentMatrix = zeros(length(SpikeTrains));
    RawCountMatrix = zeros(length(SpikeTrains));

    %Create an artificial time vector
    Ts = 0.5e-4;
    time = Ts:Ts:maxLat+10;
    
    %Try to compute all coincident events, skip if error 
    try
    
        for j = 1:length(SpikeTrains)-1
            Ej = SpikeTrains(j).Electrode;
            
            for k = j+1:length(SpikeTrains)
                Ek = SpikeTrains(k).Electrode;
                
                if((ismember(Ej,ElectrodeGroup1) && ismember(Ek,ElectrodeGroup2)) || ...
                        (ismember(Ek,ElectrodeGroup1) && ismember(Ej,ElectrodeGroup2)))
                    t1 = SpikeTrains(j).Latencies;
                    t2 = SpikeTrains(k).Latencies;
                    L1 = round(t1/Ts);
                    L2 = round(t2/Ts);
                    [C1,C2,Deltas] = SpikeSynchrony_SampleBased(L1,L2,time,MaxDifference);
                    PercentMatrix(j,k) = 2*sum(C1)/(length(t1)+length(t2));
                    RawCountMatrix(j,k) = sum(C1);
                end  
            end
            fprintf("%d%% ",round(100*j/(length(SpikeTrains)-1)))
        end

    catch
       
        disp("Error finding coincident spikes, skipping")
        continue;
        
    end
    
    fprintf("\n")
    
 %% Compute summary stats and save
    %Save the spike trains and their computed coincidence matrices
    savefile = strcat(CSVfile(1:end-5),'_UserChosenElectrodes.mat');
    save(savefile,'SpikeTrains','PercentMatrix','RawCountMatrix');
    
%% Create an excel file with some of the data
    excelSavefile = strcat(CSVfile(1:end-5),'_ExtractedData_UserChosenElectrodes.xlsx');
    Electrode_Number = [SpikeTrains.Electrode]; Electrode_Number = Electrode_Number(:);
    Number_of_spikes = zeros(size(Electrode_Number));
    for j = 1:length(SpikeTrains)
            Number_of_spikes(j)       = length(SpikeTrains(j).Latencies);
    end
    T1 = table(Electrode_Number,Number_of_spikes);
    T2 = array2table(RawCountMatrix);
    writetable(T1,excelSavefile,'WriteVariableNames',true,'Sheet','Order of Electrodes');
    writetable(T2,excelSavefile,'WriteVariableNames',false,'Sheet','Coincidence Count Matrix');
    
end
